<?php
/**
 * Plugin Name: Neurotrac Clinical Papers
 * Description: Fetch and display clinical papers by condition for a given product from clinical.neurotrac.com API.
 * Version: 1.0.0
 * Author: Neurotrac Dev
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class Neurotrac_Clinical_Papers {

    public function __construct() {
        add_shortcode('neurotrac_papers', [$this, 'render_shortcode']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_styles']);
    }

    public function enqueue_styles() {
        wp_register_style(
            'neurotrac-clinical-papers',
            plugins_url('style.css', __FILE__)
        );
        wp_enqueue_style('neurotrac-clinical-papers');
    }

    public function render_shortcode($atts) {
        $atts = shortcode_atts(array(
            'id' => 0, // product ID
        ), $atts);

        $product_id = intval($atts['id']);
        if (!$product_id) {
            return '<p>No product ID provided.</p>';
        }

        // API endpoint (change domain if needed)
        $api_url = "https://clinical.neurotrac.com/wp-json/neurotrac/v1/product?id=" . $product_id;

        // Fetch API
        $response = wp_remote_get($api_url, array('timeout' => 15));
        if (is_wp_error($response)) {
            return '<p>Could not connect to clinical papers API.</p>';
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if (empty($data) || !isset($data['papers_by_condition'])) {
            return '<p>No clinical papers found for this product.</p>';
        }

        // Build HTML output
        $output = '<div class="neurotrac-clinical-papers">';
        $output .= '<h3>Clinical Papers for ' . esc_html($data['product_title']) . '</h3>';

        foreach ($data['papers_by_condition'] as $condition) {
            $output .= '<h4 class="condition-name">Condition : ' . esc_html($condition['condition_name']) . '</h4>';
            $output .= '<ul class="clinical-papers-list">';

            foreach ($condition['papers'] as $paper) {
                $output .= '<li class="clinical-paper">';
                $output .= '<h6><a href="' . esc_url($paper['link']) . '" target="_blank">' . esc_html($paper['title']) . '</a></h6>';
                $output .= '<p>' . esc_html($paper['excerpt']) . '</p>';
                if (!empty($paper['author'])) {
                    $output .= '<p><strong>Author:</strong> ' . esc_html($paper['author']) . '</p>';
                }
                $output .= '</li>';
            }

            $output .= '</ul>';
        }

        $output .= '</div>';

        return $output;
    }
}

new Neurotrac_Clinical_Papers();
